import React, { useState, useCallback } from 'react';
import { generateAppCode } from './services/geminiService';
import { Header } from './components/Header';
import { UrlInput } from './components/UrlInput';
import { AppNameInput } from './components/AppNameInput';
import { ImageUploader } from './components/ImageUploader';
import { GenerationResult } from './components/GenerationResult';
import { GenerationInProgress } from './components/GenerationInProgress';
import { ErrorDisplay } from './components/ErrorDisplay';

// Extend the Window interface to include JSZip
declare global {
  interface Window {
    JSZip: any;
  }
}

const App: React.FC = () => {
  const [websiteUrl, setWebsiteUrl] = useState<string>('');
  const [appName, setAppName] = useState<string>('');
  const [splashImageFile, setSplashImageFile] = useState<File | null>(null);
  const [splashImageUrl, setSplashImageUrl] = useState<string>('');
  const [generatedFiles, setGeneratedFiles] = useState<Record<string, string> | null>(null);
  const [isLoading, setIsLoading] = useState<boolean>(false);
  const [error, setError] = useState<string | null>(null);

  const handleImageUpload = useCallback((file: File) => {
    setSplashImageFile(file);
    setSplashImageUrl(URL.createObjectURL(file));
  }, []);

  const handleGenerateApp = useCallback(async () => {
    if (!websiteUrl) {
      setError('Please enter a website URL.');
      return;
    }
    if (!appName) {
      setError('Please enter an application name.');
      return;
    }
    if (!splashImageFile) {
        setError('Please upload a splash screen image.');
        return;
    }

    setIsLoading(true);
    setError(null);
    setGeneratedFiles(null);

    try {
      const files = await generateAppCode(websiteUrl, appName);
      setGeneratedFiles(files);
    } catch (err) {
      console.error(err);
      setError('Failed to generate app files. The model may be unavailable. Please try again later.');
    } finally {
      setIsLoading(false);
    }
  }, [websiteUrl, appName, splashImageFile]);

  const handleDownloadZip = useCallback(async () => {
    if (!generatedFiles || !window.JSZip) {
        setError('Could not download files. JSZip library not found or no files generated.');
        return;
    }

    const zip = new window.JSZip();
    
    // Add generated files
    for (const path in generatedFiles) {
        zip.file(path, generatedFiles[path]);
    }
    
    // Add user instructions
    const readmeContent = `
# Android WebView App Project

Thank you for using the Android Website App Builder!

## Project Setup

1.  **Unzip this file.** You will find a standard Android project structure inside.
2.  **Open with Android Studio:** Open the entire unzipped folder as a project in Android Studio.
3.  **Add Your Splash Screen:**
    *   This zip file includes your splash screen image named \`UPLOADED_SPLASH_IMAGE.png\`.
    *   In Android Studio's project view (usually on the left), navigate to the \`app/src/main/res/drawable\` folder.
    *   Drag and drop \`UPLOADED_SPLASH_IMAGE.png\` from your file explorer into the \`drawable\` folder within Android Studio.
    *   A dialog will appear. **Crucially, rename the file to \`splash_screen.png\` in this dialog** before clicking OK. This will replace the placeholder image.

Once the project is set up and Gradle has synced, you can build your application.

---

## How to Build an APK (for testing and direct installation)

An APK file can be installed directly onto an Android device. This is useful for testing.

1.  In the Android Studio top menu, go to \`Build > Build Bundle(s) / APK(s) > Build APK(s)\`.
2.  Android Studio will build the app. Once finished, a notification will appear in the bottom-right corner.
3.  Click the "locate" link in the notification to find your APK file. It will be located in \`app/build/outputs/apk/debug/app-debug.apk\`.
4.  You can now copy this file to an Android device and install it.

---

## How to Build an AAB (for publishing to Google Play Store)

An Android App Bundle (AAB) is the required format for publishing new apps on the Google Play Store.

1.  In the Android Studio top menu, go to \`Build > Generate Signed Bundle / APK...\`.
2.  A dialog will appear. Select **Android App Bundle** and click Next.
3.  You will need to create a "keystore" to sign your app. This is a secure key that proves you are the developer.
    *   Click **Create new...** under the "Key store path" field.
    *   Fill in the details to create your keystore file. **Remember your passwords and store this file safely! You will need it to publish future updates.**
    *   After creating the keystore, the fields will be filled in. Click Next.
4.  Choose a destination folder for your signed bundle, select **release** as the build variant, and click **Create**.
5.  Android Studio will build your AAB file. A notification will appear when it's ready.
6.  Click the "locate" link to find your file (e.g., \`app/release/app-release.aab\`). This is the file you will upload to the Google Play Console.
`;
    zip.file('README.md', readmeContent);

    // Add the splash image
    if (splashImageFile) {
      zip.file('UPLOADED_SPLASH_IMAGE.png', splashImageFile);
    }

    // Generate and download zip
    try {
        const content = await zip.generateAsync({ type: 'blob' });
        const link = document.createElement('a');
        link.href = URL.createObjectURL(content);
        link.download = `${appName.replace(/\s/g, '_') || 'AndroidWebViewApp'}.zip`;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    } catch (err) {
        console.error(err);
        setError('Failed to create the zip file.');
    }

  }, [generatedFiles, splashImageFile, appName]);

  return (
    <div className="bg-slate-900 min-h-screen text-white font-sans flex flex-col items-center p-4 sm:p-6 md:p-8">
      <Header />
      <main className="w-full max-w-2xl mt-8">
        <div className="bg-slate-800/50 rounded-2xl shadow-2xl p-6 sm:p-8 backdrop-blur-sm border border-slate-700">
          
          {isLoading ? (
            <GenerationInProgress />
          ) : (
            <>
              <div className="space-y-8">
                <UrlInput value={websiteUrl} onChange={setWebsiteUrl} />
                <AppNameInput value={appName} onChange={setAppName} />
                <ImageUploader onImageUpload={handleImageUpload} imageUrl={splashImageUrl} />

                <div className="pt-4 border-t border-slate-700">
                  <button
                    onClick={handleGenerateApp}
                    disabled={!websiteUrl || !appName || !splashImageFile}
                    className="w-full bg-indigo-600 hover:bg-indigo-500 disabled:bg-indigo-900 disabled:text-slate-400 disabled:cursor-not-allowed text-white font-bold py-3 px-4 rounded-lg text-lg transition-all duration-300 transform hover:scale-105 flex items-center justify-center shadow-lg"
                  >
                    Generate App Files
                  </button>
                </div>
              </div>

              {error && <ErrorDisplay message={error} />}
              
              {generatedFiles && (
                <GenerationResult 
                  files={Object.keys(generatedFiles)} 
                  onDownload={handleDownloadZip} 
                />
              )}
            </>
          )}
        </div>
        <footer className="text-center mt-12 text-slate-500 text-sm">
            <p>Powered by Google Gemini. App files are generated for Android Studio.</p>
        </footer>
      </main>
    </div>
  );
};

export default App;
